#ifndef DM_SERIAL_H
#define DM_SERIAL_H

#if defined (DMSWITCH_BUILD)
#include <npp/npp.h>                    // for Task
#include "CSerialInterrupts.h"
#include "dm_sharedTask.h"
#endif
#include "dm_nutypedefs.h"

#define DM_HDBASET_UART_FUNC_SEL_NONE   0
#define DM_HDBASET_UART_FUNC_SEL_DMNET  1
#define DM_HDBASET_UART_FUNC_SEL_SERIAL 2
extern UINT8 DmHDBaseTUartFuncSelGet(UINT8 dm_idx /* 0-based index of DM port */);
extern void DmHDBaseTUartFuncSelSet(UINT8 dm_idx /* 0-based index of DM port */ , UINT8 func /* DM_UART_FUNC_* */,UINT8 bEnable);

// Comspec definitions for type 12 packet
#define COMSPEC_300     0x0000
#define COMSPEC_600	0x0001
#define COMSPEC_1200	0x0002
#define COMSPEC_2400	0x0003
#define COMSPEC_4800	0x0004
#define COMSPEC_9600	0x0005
#define COMSPEC_19200	0x0006
#define COMSPEC_38400	0x0007
#define COMSPEC_1800	0x0080
#define COMSPEC_3600	0x0081
#define COMSPEC_7200	0x0082
#define COMSPEC_14400	0x0083
#define COMSPEC_28800	0x0084
#define COMSPEC_57600	0x0085
#define COMSPEC_115200	0x0086
#define COMSPEC_230400	0x0087
#define COMSPEC_BRMASK	0x0087

#define COMSPEC_PNONE	0x0000
#define COMSPEC_PEVEN	0x0008
#define COMSPEC_PZERO	0x0010
#define COMSPEC_PODD	0x0018
#define COMSPEC_PMASK	0x0018

#define COMSPEC_7BITS	0x0000
#define COMSPEC_8BITS	0x0020
#define COMSPEC_BMASK	0x0020

#define COMSPEC_1STOP	0x0000
#define COMSPEC_2STOP	0x0040
#define COMSPEC_SMASK	0x0040

#define COMSPEC_RS232	0x0000
#define COMSPEC_RS422	0x0100
#define COMSPEC_RS485	0x2100
#define COMSPEC_RSMASK	0x2100

#define COMSPEC_CTS	0x0200
#define COMSPEC_RTS	0x0400
#define COMSPEC_XOFFT   0x0800
#define COMSPEC_XOFFR   0x1000
#define COMSPEC_RCTS 	0x4000   // Report CTS

#define COMSPEC_PROTOCOL_MASK  0x2100
#define COMSPEC_PROTOCOL_RS232 0x0000
#define COMSPEC_PROTOCOL_RS422 0x0100
#define COMSPEC_PROTOCOL_RS485 0x2100

#define XON     0x11
#define XOFF    0x13

#define UART_CRESNET_PARITY_NO   0x0000  /* Parity Disable */
#define UART_CRESNET_PARITY_ODD  0x0002  /* Odd Parity */
#define UART_CRESNET_PARITY_EVEN 0x0006  /* Even Parity */

/* UART Hardware Flow Control */
#define UART_CRESNET_HW_FLOW_CONTROL_NONE      0x0000  /* HFC Disable */
#define UART_CRESNET_HW_FLOW_CONTROL_RTS_CTS   (COMSPEC_CTS | COMSPEC_RTS)  /* CTS and RTS Enable */
#define UART_CRESNET_HW_FLOW_CONTROL_CTS   COMSPEC_CTS  /* CTS Enable */
#define UART_CRESNET_HW_FLOW_CONTROL_RTS   COMSPEC_RTS  /* RTS Enable */

#define SERIAL_TASK_PERIOD 10 //milliseconds
#if defined (DMSWITCH_BUILD)
#define SERIAL_TASK_STACK_SIZE (1024*4)
#else
#define SERIAL_TASK_STACK_SIZE 400 //240
#endif
#define MAX_SERIAL_PACKET 120
#define MAX_SERIAL_TX 10

#define SERIAL_TASK_RX_SETTLE_POLL_MS           5
#define SerialPortSettleTime(a)                 (a / SERIAL_TASK_RX_SETTLE_POLL_MS)
#define DM_SERIAL_PORT_SETTLE_TIME              50

#define DEFAULT_TXRX_BUFFER_SIZE 128
#define SERIAL_DATA_INIT_STATE 0xff

#define   SERIAL_PORT_COMA  0
#define   SERIAL_PORT_COMB  1
#define   SERIAL_PORT_COMC  2
#define   SERIAL_PORT_COMD  3
#define   SERIAL_PORT_COME  4
#define   SERIAL_PORT_COMF  5

#define   NEW_COMSPEC_NONE    0
#define   WRITE_NEW_COMSPEC   1
#define   WRITING_NEW_COMSPEC 2


/* Report the comspec back to the control system for 3 series
    100  150  200  250  300  350      A	  From Driver   Baud		300 ... 57600, 115=115200
    101  151  201  251  301  351      A	  From Driver	Data Bits	7, 8
    102  152  202  252  302  352      A	  From Driver	Parity		'N', 'E', 'O'
    103  153  203  253  303  353      A	  From Driver	Stop Bits	1, 2
    104  154  204  254  304  354      A	  From Driver	Protocol        232:  0, 422:  1, 485:  2
    105  155  205  255  305  355      A	  From Driver	HW Handshake    RTS:  1, CTS:  2, RTS/CTS:  3
    106  156  206  256  306  356      A	  From Driver	SW Handshake    XONT & XONR:  1, XONT:  2,  XONR:  3*/
#define TLDM_COMPORT_JOIN_NUM_OFFSET      50 // The joins for the COMPORTS start at 100 and are offset by 50
#define TLDM_COMPORT_SUPPORTS_RS232       0
#define TLDM_COMPORT_SUPPORTS_RS422       1
#define TLDM_COMPORT_SUPPORTS_RS485       2
#define TLDM_COMPORT_USE_RTS              1
#define TLDM_COMPORT_USE_CTS              2
#define TLDM_COMPORT_XONT_XONR_ENABLE     1
#define TLDM_COMPORT_XONT_ENABLED         2
#define TLDM_COMPORT_XONR_ENABLED         3

enum
{
  TLDM_COM_PORT_BAUD_RATE_REPORTING_JOIN=100,     // Analog Sent to Logic
  TLDM_COM_PORT_DATA_BITS_REPORTING_JOIN,         // Analog Sent to Logic
  TLDM_COM_PORT_PARITY_REPORTING_JOIN,            // Analog Sent to Logic
  TLDM_COM_PORT_STOP_BITS_REPORTING_JOIN,         // Analog Sent to Logic
  TLDM_COM_PORT_PROTOCOL_REPORTING_JOIN,          // Analog Sent to Logic
  TLDM_COM_PORT_HW_REPORTING_JOIN,                // Analog Sent to Logic
  TLDM_COM_PORT_SW_REPORTING_JOIN                 // Analog Sent to Logic
};

struct SerialPortCfg
{
    UINT8    Channel;/**< Which serial port channel */    
    UINT8    sendParam; /**< MCU uart instance */
    UINT32   taskPrio;/**< serial tasks priority */
    UINT8*   pTxBuf;/**< pointer to buffer of characters to be sent to serial port */
    UINT8*   pRxBuf;/**< pointer to buffer of characters received from serial port */
    UINT32   SerialOutputJoin; /**< serial output join# for this channel */
    UINT32   RTS_DigitalInputJoin;/**< RTS digital input join# for this channel */
    UINT32   CTS_DigitalOutputJoin;/**< CTS digital output join# for this channel */
    UINT32   FeedbackSettleTime;            /**< of ticks for serial join feedback settling time (set zero to bypass) */
    UINT16   TxRxBufferSize;/**< tx and rx buffer size */
    UINT16   TxRxHighWaterMark;/**< number of unprocessed bytes in rx queue that flag an error */
    UINT16   TxRxReleaseHighWaterMark;/**< number of unprocessed bytes in rx queue that frees up rx stop */
    BOOL     ExternalUART; /**< external uart supported */
    BOOL     UseTxCompInt;/**< Use transfer complete interrupt */
    UINT16   TaskStackSize;/**< stack size for TX and RX tasks, changes based on priority */
	BOOL     CreateInternalTask; /**< create internal task. If false, requires external management of task. */
    BOOL     IsDynamicUartConfig; /**< Will the UART be reconfigured dynamically */
#if defined (DMSWITCH_BUILD)    
    UINT16   bStream;
#endif    
};
extern const struct SerialPortCfg SerialCfg[];
// the hardware library is compiled as C and therefore should not include the class
#ifdef __cplusplus
#if defined (DMSWITCH_BUILD)
class SerialPort;

class CSerialPortTask : public Task
{
    public:
        CSerialPortTask(char* TaskName, SharedTask * pSharedTask, UINT8 TaskPrio, UINT32 stacksize );
        ~CSerialPortTask(){};

        SharedTask * m_pSharedTask;

    protected:
        virtual VOID Entry();
};
// On the 6x4 this is driven off the shared FPGA interrupt so it is not a LISR per se...
class CDMSerialPortLISR
{
    public:

        EXAR_16C684_PORT *exar_port ;
        SerialPort *m_pOwner;

        CDMSerialPortLISR(SerialPort *pOwner, EXAR_16C684_PORT *Exar_Port );
        void WriteTxBuffer(unsigned char byt) { exar_port->THR_RG = byt; }
        void        EntryRoutine();     // Actual ISR.

};
#endif //#if defined (DMSWITCH_BUILD)
class SerialPort
{
public:
    SerialPort(const SerialPortCfg *pSerialCfg);
    ~SerialPort();

    UINT32 m_pSharedTask;
#if defined (DMSWITCH_BUILD)    
    CSerialPortTask * m_pCSerialPortTask;
    CDMSerialPortLISR      * m_pLISR;
#else
	UINT32 m_TaskTCB;
#endif    
    
    INT32 SerialTXThread(void);
    INT32 SerialRXThread(void);    
    void AssertRTS( BOOL state);
    BOOL GetCTS( void );
    void TxSerialData(UINT8 *data, UINT32 count, UINT32 inst);
    void TxSerialDataINT(UINT32 uartNum, UINT8 data);
    void UartConfig(UINT32 inst, UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol);
    void ReceiveTxData( char *ptr, UINT32 count );
    void TxLoadNextChar( void );
    UINT32 TxRxIsr( BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty);
    void ReceiveComspec( UINT8 *ptr );
    void SetNewComParameters(UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol);
    void EnableHwHandShake(BOOL enable);
    BOOL ReturnHwHandShake(void);
    void GetHwConfig(UINT32 *pbaud, UINT32 *pstopbits, UINT32 *pparity, UINT32 *pwordlength);
    void SetRTSToDigitalJoinState( BOOL state);
    void SetNewComSpec( void );
    void SetCommParameters( UINT8 comSpec );
    void CreateExternalUartMsgQueue(void);
    void DestroyExternalUartMsgQueue(void);
    void ServiceExternalUart(void);
    void SetNewProtocol(UINT16 Protocol);
    void EnableTransferComplete(void);
    void EnableReceive(void);
    void EnableTxInt(void);
    void ReportComSpec(UINT32 Baudrate, UINT8 StopBits, UINT8 Parity, UINT8 DataBits);
    void ReportOverflow(void);
    void ClearOverflowFlag(void) {m_OverFlowFlag = FALSE;};
    void Reconfigure(BOOL bEnable);
    void DisableHardware(void);
    void PrintDebug(void);
    UINT8 GetChannel( void )
    {
       return m_pSerialCfg->Channel;
    }
#if defined (DMSWITCH_BUILD)     
    UINT8 GetStream( void )
    {
       return m_pSerialCfg->bStream;
    }
#endif    
    UINT8* InitTxBufferPointer(void);
    UINT8* InitRxBufferPointer(void);
    void DeInitTxBufferPointer(void);
    void DeInitRxBufferPointer(void);

	friend INT32 SerialPortDeviceProcessInput(UINT32 inst, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes);

private:

    UINT32 GetRxCount( void );

    UINT8    m_NewComSpec[4];/**< copy of most recent comspec */
    UINT8    m_ChangingComSpec;/**< Set when comspec is changing */
    UINT8    m_OverFlowFlag; /**< Flag to prevent sending rx buffer overflow error more than once */
    UINT8    m_CurrentCTS;/**< last read CTS signal level */
    UINT8    m_EnableReportCTS;/**< flag to track if CTS state should be reported to SIMPL */
    UINT8    m_StoppedTX;/**< flag to track if tx stopped by flow control */
    UINT8    m_StoppedRX;/**< flag to track if rx stopped by flow control */
    UINT8    m_HandshakeCTS;/**< CTS handshake flag received in com spec */
    UINT8    m_HandshakeRTS;/**< RTS handshake flag received in com spec */
    UINT8    m_HandshakeXOFFT;/**< XOFF transmit handshake flag received in com spec */
    UINT8    m_HandshakeXOFFR;/**< XOFF receive handshake flag received in com spec */
    UINT8    m_CurrentXOFFT;/**< last XOFF state */
    UINT8    m_PaceTX;/**< character tx pacing flag */
    UINT8    m_ActiveTX;/**< actively txing in interrupt */
    UINT8    m_Stream;  /**< stream to send output to.*/
    UINT8    m_UseTxCompInt; /**< Use TX complete interrupt to load data for transmit*/
    const SerialPortCfg *m_pSerialCfg;/**< configuration tables */
    UINT8*   m_pTxBuf;/**< pointer to buffer of characters to be sent to serial port */
    UINT8*   m_pRxBuf;/**< pointer to buffer of characters received from serial port */
    UINT32   m_BaudRate;/**< baudrate */
    UINT16   m_PaceCount;/**< character tx pacing count */
    UINT16   m_stopbits;/**< stop bits */
    UINT16   m_parity;/**< parity */
    UINT16   m_wlenght;/**< word length */
    UINT16   m_Protocol;  /**< active protocol (rs232, 422, 485)*/
    UINT32   m_Txin;/**< tx buffer in index */
    UINT32   m_Txout;/**< tx buffer out index */
    UINT32   m_Rxin;/**< rx buffer in index */
    UINT32   m_Rxout;/**< rx buffer out index */

    BOOL     m_bTxEnable; /**< control if tx allowed (TRUE) or short circuit (FALSE) */

    INT32 m_iSettleTimer;
};

SerialPort *GetSerialPort(UINT8 Channel);
void CreateSerialPorts( UINT8 bNumSp, const SerialPortCfg *pSerialCfg, BOOL bUseSingleSharedTask );
extern SerialPort *GetSerialPortfromStream(UINT8 bStream);

#endif // __cplusplus

void SerialPortReceiveTxData( UINT8 Channel, char *ptr, UINT32 count );
void ProcessDigitalRTSJoin( UINT8 Channel, BOOL state );
void SerialPortReceiveComspec(UINT8 *ptr, UINT8 Channel );
UINT32 SerialPortIsr(UINT32 Channel, BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty);

// Serial Errors
typedef enum
{
  ERR_SERIAL_RX_BUFFER_OVERFLOW,        // 0 serial recieve buffer overflow
  ERR_SERIAL_BUFFER_OVERFLOW_NULL_PTR,  // 1 serial recieve buffer overflow
  ERR_SERIAL_NO_MEMORY,                 // 2 no memory to initilaze serial port
  ERR_SERIAL_NO_CONFIG_TABLE,           // 3
  ERR_SERIAL_TASK_NO_EVENT,             // 4
  ERR_SERIAL_CRESTNET_SLAVE_QUEUE_FULL, // 5 no Mem blocks to send rx serial to cresnet
  ERR_SERIAL_UNSUPPORTED_PROTOCOL,      // 6 unsupported serial protocol
  ERR_SERIAL_INVALID_CHANNEL_COMSPEC_RX,// 7 no serial port on slot # in comspec rx'd
  ERR_SERIAL_INVALID_THREAD_ARG_TYPE,   // 8 Invalid thread argument type
  ERR_SERIAL_HDBASET_DRIVER_NOT_INITIALIZED,   // 9 Invalid thread argument type
  ERR_SERIAL_HDBASET_POLL_DEVICE_UART_ERROR // 10 DMNET UART stop poll error

}ERR_SERIAL;
#if defined (DMSWITCH_BUILD)
extern int SendSerialCmd(int ignore, char * cmd);
extern int SetCommParameters(int ignore, char * cmd);
extern int SetUartHWHandshake(int ignore, char *cmd);
#else
extern INT32 SendSerialCmd(UINT32 ignore, char * cmd);
extern INT32 SetCommParameters(UINT32 ignore, char * cmd);
extern INT32 SetUartHWHandshake(UINT32 ignore, char *cmd);

#endif

INT32 SerialPortDeviceProcessInput(UINT32 Channel, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes);
#define SERIALPORT_NULLPTR_RECALL_DELAY 1 //SHAREDTASK_MIN_SLEEPTIME
#ifndef SERIAL_TASK_PRIORITY
#if defined (DMSWITCH_BUILD)
#define SERIAL_TASK_PRIORITY 40
#else
#define SERIAL_TASK_PRIORITY 0
#endif
#endif
typedef enum
{
    SERIAL_PORT_RX_THREAD_TYPE,           /**< rx thread type */
    SERIAL_PORT_TX_THREAD_TYPE,           /**< tx thread type */
    SERIAL_PORT_MAX_THREAD_TYPE           /**< max thread type */
}SERIAL_PORT_THREAD_TYPE;
  
typedef struct
{
    SERIAL_PORT_THREAD_TYPE eThreadType;  /**< rx/tx thread type */
    UINT8 Channel;                        /**< Which serial port channel */

}SERIAL_PORT_THREAD_ARGUMENTS;

#endif
